/*
 * TagEvent.h
 *
 *  Created on: 21.07.2011
 *      Author: stefan.detter
 */

#ifndef TAGEVENT_H_
#define TAGEVENT_H_

#include <QObject>
#include <QMetaType>
#include <QVariant>
#include <QDateTime>
#include <QDataStream>

class TagEvent
{
public:
	TagEvent();
	TagEvent(const QString& readerId, const QString& readerName, const QString& readerLocation,
			const QString& type, const uint tagTypeId, const QString& tagId, const QByteArray& tagIdBytes);
	TagEvent(const TagEvent& other);
	virtual ~TagEvent();

	const QString& 		readerId() const;
	const QString& 		readerName() const;
	const QString& 		readerLocation() const;
	const QString&		tagType() const;
	const quint32		tagTypeId() const;
	const QString& 		tagId() const;
	const QByteArray&	tagIdBytes() const;
    const QString& 		visualTagId() const;
    const QDateTime&	readTime() const;
	const quint64 		readCount() const;

	void 					addInformation(const uint infoType, const QVariant& info);
	const QList<QVariant>	getInformations() const;
	bool 					hasInformation(const uint infoType) const;
	QVariant 				getInformation(const uint infoType) const;

	void increasReadCount(ulong readCount);

    void setVisualTagId(const QString &visualTagId);

private:
    QString 			m_readerId;
    QString 			m_readerName;
	QString 			m_readerLocation;

	QDateTime			m_readTime;
	quint64				m_readCount;

	QString				m_tagType;
	uint				m_tagTypeId;
	QString 			m_tagId;
	QByteArray 			m_tagIdBytes;
    QString 			m_visualTagId;

	QMap<uint,QVariant>	m_informations;

	friend QDataStream & operator<< (QDataStream& stream, const TagEvent& image);
	friend QDataStream & operator>> (QDataStream& stream, TagEvent& image);
};
Q_DECLARE_METATYPE(TagEvent)


typedef enum {
	UNKNOWN = 0,
	RSSI_INFORMATION = 1,
	MEMORY_INFORMATION = 2,
	TRIGGER_INFORMATION = 3,
	ANTENNA_INFORMATION = 4,
	FREQUENCY_INFORMATION = 5,
	HANDLE_INFORMATION = 6,
	STATE_INFORMATION = 7,
	BATTERY_INFORMATION = 8,
	PC_INFORMATION = 9,
    MESSAGEID_INFORMATION = 10,
    RN16_INFORMATION = 11,
    NXP_BRANDID_INFORMATION = 12,
    GENERIC_INFORMATION = 100,
} INFORMATION_TYPE;

class TagEventInformation
{
public:
	TagEventInformation(const quint32 type, const QString& typeString);
	TagEventInformation();
	TagEventInformation(const TagEventInformation& other);
	virtual ~TagEventInformation();

	const quint32	type() const;
	const QString& 	typeString() const;
protected:
	quint32				m_type;
	QString 			m_typeString;
};


/*****************************************************************************************************************************************************/
class RSSI_Information : public TagEventInformation
{
public:
	static const uint 		id = RSSI_INFORMATION;

	typedef struct{
		QString 	name;
		qint32		minimum;
		qint32		maximum;
		qint32		value;
		QString 	unit;
	} CHANNEL;

	RSSI_Information(const uchar percent);
	RSSI_Information();
	RSSI_Information(const RSSI_Information& other);
	virtual ~RSSI_Information();

	quint8 	percent() const;

	void 					addChannel(const QString& name, const qint32 minimum, const qint32 maximum, const qint32 value, const QString& unit);
	const QList<CHANNEL>& 	getChannels() const;

private:
	quint8 				m_percent;
	QList<CHANNEL>		m_channels;

	friend QDataStream & operator<< (QDataStream& stream, const RSSI_Information& image);
	friend QDataStream & operator>> (QDataStream& stream, RSSI_Information& image);
};
Q_DECLARE_METATYPE(RSSI_Information)


/*****************************************************************************************************************************************************/
class MEMORY_Information : public TagEventInformation
{
public:
	static const uint 		id = MEMORY_INFORMATION;

	MEMORY_Information(const QString& memBankName, const quint32 memBank, const quint32 memAddr, const QByteArray& memData);
	MEMORY_Information();
	MEMORY_Information(const MEMORY_Information& other);
	virtual ~MEMORY_Information();

	const QString&	 	memBankName() const;
	const quint32 		memBank() const;
	const quint32 		memAddr() const;
	const QByteArray& 	memData() const;

private:
	QString 	m_memBankName;
	quint32	 	m_memBank;
	quint32 	m_memAddr;
	QByteArray 	m_memData;

	friend QDataStream & operator<< (QDataStream& stream, const MEMORY_Information& image);
	friend QDataStream & operator>> (QDataStream& stream, MEMORY_Information& image);
};
Q_DECLARE_METATYPE(MEMORY_Information)


/*****************************************************************************************************************************************************/
class TRIGGER_Information : public TagEventInformation
{
public:
	static const uint 		id = TRIGGER_INFORMATION;

	TRIGGER_Information( const quint32 triggerId, const QString& triggerName);
	TRIGGER_Information();
	TRIGGER_Information(const TRIGGER_Information& other);
	virtual ~TRIGGER_Information();

	const quint32	triggerId() const;
	const QString& 	triggerName() const;

private:
	quint32		m_triggerId;
	QString 	m_triggerName;

	friend QDataStream & operator<< (QDataStream& stream, const TRIGGER_Information& image);
	friend QDataStream & operator>> (QDataStream& stream, TRIGGER_Information& image);
};
Q_DECLARE_METATYPE(TRIGGER_Information)


/*****************************************************************************************************************************************************/
class ANTENNA_Information : public TagEventInformation
{
public:
	static const uint 		id = ANTENNA_INFORMATION;

	ANTENNA_Information( const quint32 antennaId );
	ANTENNA_Information();
	ANTENNA_Information(const ANTENNA_Information& other);
	virtual ~ANTENNA_Information();

	const quint32	antennaId() const;

private:
	quint32		m_antennaId;

	friend QDataStream & operator<< (QDataStream& stream, const ANTENNA_Information& image);
	friend QDataStream & operator>> (QDataStream& stream, ANTENNA_Information& image);
};
Q_DECLARE_METATYPE(ANTENNA_Information)


/*****************************************************************************************************************************************************/
class FREQUENCY_Information : public TagEventInformation
{
public:
	static const uint 		id = FREQUENCY_INFORMATION;

	FREQUENCY_Information( const quint32 frequency );
	FREQUENCY_Information();
	FREQUENCY_Information(const FREQUENCY_Information& other);
	virtual ~FREQUENCY_Information();

	const quint32	frequency() const;

private:
	quint32		m_frequency;

	friend QDataStream & operator<< (QDataStream& stream, const FREQUENCY_Information& image);
	friend QDataStream & operator>> (QDataStream& stream, FREQUENCY_Information& image);
};
Q_DECLARE_METATYPE(FREQUENCY_Information)


/*****************************************************************************************************************************************************/
class HANDLE_Information : public TagEventInformation
{
public:
	static const uint 		id = HANDLE_INFORMATION;

	HANDLE_Information( const QByteArray& handle );
	HANDLE_Information();
	HANDLE_Information(const HANDLE_Information& other);
	virtual ~HANDLE_Information();

	const QByteArray	handle() const;

private:
	QByteArray		m_handle;

	friend QDataStream & operator<< (QDataStream& stream, const HANDLE_Information& image);
	friend QDataStream & operator>> (QDataStream& stream, HANDLE_Information& image);
};
Q_DECLARE_METATYPE(HANDLE_Information)


/*****************************************************************************************************************************************************/
class STATE_Information : public TagEventInformation
{
public:
	static const uint 		id = STATE_INFORMATION;

	STATE_Information( const quint16 state, const QString& stateName );
	STATE_Information();
	STATE_Information(const STATE_Information& other);
	virtual ~STATE_Information();

	const quint16	state() const;
	const QString& 	stateName() const;

private:
	quint16		m_state;
	QString 	m_stateName;

	friend QDataStream & operator<< (QDataStream& stream, const STATE_Information& image);
	friend QDataStream & operator>> (QDataStream& stream, STATE_Information& image);
};
Q_DECLARE_METATYPE(STATE_Information)


/*****************************************************************************************************************************************************/
class BATTERY_Information : public TagEventInformation
{
public:
	static const uint 		id = BATTERY_INFORMATION;

	BATTERY_Information( const quint8 battery );
	BATTERY_Information();
	BATTERY_Information(const BATTERY_Information& other);
	virtual ~BATTERY_Information();

	const quint8	battery() const;

private:
	quint8		m_battery;

	friend QDataStream & operator<< (QDataStream& stream, const BATTERY_Information& image);
	friend QDataStream & operator>> (QDataStream& stream, BATTERY_Information& image);
};
Q_DECLARE_METATYPE(BATTERY_Information)


/*****************************************************************************************************************************************************/
class PC_Information : public TagEventInformation
{
public:
	static const uint 		id = PC_INFORMATION;

	PC_Information( const QByteArray& pc );
	PC_Information();
	PC_Information(const PC_Information& other);
	virtual ~PC_Information();

	const QByteArray	pc() const;

private:
	QByteArray		m_pc;

	friend QDataStream & operator<< (QDataStream& stream, const PC_Information& image);
	friend QDataStream & operator>> (QDataStream& stream, PC_Information& image);
};
Q_DECLARE_METATYPE(PC_Information)

/*****************************************************************************************************************************************************/
class MESSAGEID_Information : public TagEventInformation
{
public:
	static const uint 		id = MESSAGEID_INFORMATION;

	MESSAGEID_Information( const quint8 messageId );
	MESSAGEID_Information();
	MESSAGEID_Information(const MESSAGEID_Information& other);
	virtual ~MESSAGEID_Information();

	const quint8	messageId() const;

private:
	quint8		m_messageId;

	friend QDataStream & operator<< (QDataStream& stream, const MESSAGEID_Information& image);
	friend QDataStream & operator>> (QDataStream& stream, MESSAGEID_Information& image);
};
Q_DECLARE_METATYPE(MESSAGEID_Information)

/*****************************************************************************************************************************************************/
class RN16_Information : public TagEventInformation
{
public:
    static const uint 		id = RN16_INFORMATION;

    RN16_Information( const QByteArray& rn16 );
    RN16_Information();
    RN16_Information(const RN16_Information& other);
    virtual ~RN16_Information();

    const QByteArray	rn16() const;

private:
    QByteArray		m_rn16;

    friend QDataStream & operator<< (QDataStream& stream, const RN16_Information& image);
    friend QDataStream & operator>> (QDataStream& stream, RN16_Information& image);
};
Q_DECLARE_METATYPE(RN16_Information)

/*****************************************************************************************************************************************************/
class NXP_BRANDID_Information : public TagEventInformation
{
public:
    static const uint 		id = NXP_BRANDID_INFORMATION;

    NXP_BRANDID_Information( const QByteArray& brandId );
    NXP_BRANDID_Information();
    NXP_BRANDID_Information(const NXP_BRANDID_Information& other);
    virtual ~NXP_BRANDID_Information();

    const QByteArray	brandId() const;

private:
    QByteArray		m_nxpBrandId;

    friend QDataStream & operator<< (QDataStream& stream, const NXP_BRANDID_Information& image);
    friend QDataStream & operator>> (QDataStream& stream, NXP_BRANDID_Information& image);
};
Q_DECLARE_METATYPE(NXP_BRANDID_Information)

/*****************************************************************************************************************************************************/
class GENERIC_Information : public TagEventInformation
{
public:
	static const uint 		id = GENERIC_INFORMATION;

	GENERIC_Information( const uint infoId, const QString description, const QString value );
	GENERIC_Information();
	GENERIC_Information(const GENERIC_Information& other);
	virtual ~GENERIC_Information();

	const uint		infoId() const;
	const QString	description() const;
	const QString	value() const;

private:
	uint 		m_infoId;
	QString		m_description;
	QString		m_value;

	friend QDataStream & operator<< (QDataStream& stream, const GENERIC_Information& image);
	friend QDataStream & operator>> (QDataStream& stream, GENERIC_Information& image);
};
Q_DECLARE_METATYPE(GENERIC_Information)


QDataStream & operator<< (QDataStream& stream, const TagEvent& event);
QDataStream & operator>> (QDataStream& stream, TagEvent& event);
QDataStream & operator<< (QDataStream& stream, const RSSI_Information& info);
QDataStream & operator>> (QDataStream& stream, RSSI_Information& info);
QDataStream & operator<< (QDataStream& stream, const RSSI_Information::CHANNEL& channel);
QDataStream & operator>> (QDataStream& stream, RSSI_Information::CHANNEL& channel);
QDataStream & operator<< (QDataStream& stream, const MEMORY_Information& info);
QDataStream & operator>> (QDataStream& stream, MEMORY_Information& info);
QDataStream & operator<< (QDataStream& stream, const TRIGGER_Information& info);
QDataStream & operator>> (QDataStream& stream, TRIGGER_Information& info);
QDataStream & operator<< (QDataStream& stream, const ANTENNA_Information& info);
QDataStream & operator>> (QDataStream& stream, ANTENNA_Information& info);
QDataStream & operator<< (QDataStream& stream, const FREQUENCY_Information& info);
QDataStream & operator>> (QDataStream& stream, FREQUENCY_Information& info);
QDataStream & operator<< (QDataStream& stream, const HANDLE_Information& info);
QDataStream & operator>> (QDataStream& stream, HANDLE_Information& info);
QDataStream & operator<< (QDataStream& stream, const STATE_Information& info);
QDataStream & operator>> (QDataStream& stream, STATE_Information& info);
QDataStream & operator<< (QDataStream& stream, const BATTERY_Information& info);
QDataStream & operator>> (QDataStream& stream, BATTERY_Information& info);
QDataStream & operator<< (QDataStream& stream, const PC_Information& info);
QDataStream & operator>> (QDataStream& stream, PC_Information& info);
QDataStream & operator<< (QDataStream& stream, const MESSAGEID_Information& image);
QDataStream & operator>> (QDataStream& stream, MESSAGEID_Information& image);
QDataStream & operator<< (QDataStream& stream, const GENERIC_Information& image);
QDataStream & operator>> (QDataStream& stream, GENERIC_Information& image);



class __init_TagEvent
{
public:
	__init_TagEvent()
	{
		qRegisterMetaType<TagEvent>("TagEvent");
	}
};



#endif /* TAGEVENT_H_ */
